/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * olist.c
 *
 * output a list file
 */
#include <stdio.h>                      
#include <memory.h>
#include <string.h>
#include "utype.h"
#include "umem.h"
#include "asm.h"
#include "input.h"
#include "gencode.h"
#include "olist.h"
#include "sections.h"
#include "interp.h"

#define MODE8051 4
#define TYPE8051BIT 0x101
extern long lconfig;

extern char *typelist[];
extern BOOL prm_xref;
extern BYTE buf[BUFLEN];
extern HASHREC **hashtable;
extern BOOL prm_symdump;
extern int level, macrolevel;
extern FILE *listfile;
extern int pass;
extern int opcodesize;
extern BYTE pcbuf[BUFLEN];
extern LIST *sections;
extern BOOL ifskip;
extern int lineno[INCLUDELEVELS];
extern long org;
extern int currentphifg;
extern BOOL phiused;
extern procname[];
extern int version;
extern LIST *incfiles;
 extern int macstyle,ifstyle;
extern int macrolevel;
extern char fixlistvals[];
extern int fixlistnum;

BOOL setequ;
BYTE setequval[12];

static BOOL listenabled = TRUE;
static BOOL pagenabled = TRUE;

static int xrefpos;
static uint pageline = PAGEMAX *2;
static uint pagenum = 1;
static char ut[] = "Untitled";
static char *subttl=0, *title = ut;
static char nosection[] = "None";
static char mlabel[] = "label";
static char moperand[] = "operand";
static char mregs[] = "reg list";
static char mequ[] = "equ";
static char mset[] = "set";
static char mstruct[] ="struct";
static char mmacro[] = "macro";
static char mextern[] = "E EbEwEl";
static char null[]="";

/* options */
void liston(BOOL flag)
{
  listenabled = flag;
	dolist();
}
void pageon(BOOL flag)
{
  pagenabled = flag;
	pageline = PAGEMAX +1;
	dolist();
}
/* do a page break */
void pagebreak(BOOL needed)
{
  if ((needed ||++pageline >= PAGEMAX) && pagenabled) {
		int i;
		char buf[BUFLEN];
		int oldcolor = currentphifg;
		fputc(12,listfile);
		phifg(DKGREEN,listfile);
		sprintf(buf,"%s - %s version %d.%02d",title,procname,version/100, version %100);
		fprintf(listfile,buf);
		for (i=strlen(buf); i < 68; i++)
			fputc(' ', listfile);
		fprintf(listfile,"PAGE %3d\n",pagenum++);
		pageline = 2;
		if (subttl) {
			fprintf(listfile,"%s\n",subttl);
			pageline++;
		}
		phifg(oldcolor,listfile);
		fputc('\n', listfile);
	}
}
/* change the number of lines/page */
void pagelines(EXPRESSION *exp)
{
  if (!numericExpression(exp) || exp->islabel || !exp->isdef || exp->isextern || exp->relmode) {
		Error("Invalid expression");
		dolist();
	}
	else {
		int i;
  	dolist();
		if (listfile && listenabled) {
			for (i=0; i < exp->x.value; i++) {
				pagebreak(FALSE);
				fputc('\n', listfile);
			}
		}
	}
}
/* set a title
 */
void setitle(char *string)
{
	if (title != ut)
		DeallocateMemory(title);
	if (strlen(string)) {
		title = AllocateMemory(strlen(string)+1);
		strcpy(title,string);
	}
	else
		title = ut;
	dolist();
}
/* set a subtitle */
void setsubtitle(char *string)
{
  DeallocateMemory(subttl);
	subttl = 0;
	if (strlen(string)) {
		subttl = AllocateMemory(strlen(string)+1);
		strcpy(subttl,string);
	}
	dolist();
}		
/* list a line */
void dolist2(void)
{
	if (ifskip)
		opcodesize = 0;
  if (pass == 2 && listfile && listenabled) {
		int i;
		phifg(DKBLUE,listfile);
		pagebreak(FALSE);

		/* lineno and macro level info and org */
		if (level)
			fprintf(listfile,"#%2d ",level);
		else
			fprintf(listfile, "    ");
		if (macrolevel)
			fprintf(listfile,"%2d", macrolevel);
		else
			fprintf(listfile,"  ");
		fprintf(listfile,"%5d: %06lX = ",lineno[level],org);

		/* if it is an equate set the value */
		if (setequ) {
			opcodesize = 0;
			setequ = FALSE;
				fprintf(listfile,"  <%08lX>            ", *((long *)setequval));
		}
		else {
			/* otherwise dump opcode bytes */
			for (i=0; i < ((opcodesize < 13) ? opcodesize : 12) ; i++)
				fprintf(listfile,"%02X",pcbuf[i]);
			if (fixlistnum) {
				i++;
				fputc(fixlistvals[0],listfile);
				if (fixlistnum>1)
					fputc(fixlistvals[1],listfile);
				else
					fputc(' ',listfile);
			}
			for (; i < 13; i++)
				fprintf(listfile,"  ");
		}
		if (phiused)
			phifg(0,listfile);
		/* dump the line */
		fprintf(listfile,"  %s",buf);
		/* if we have more opcode bytes dump them on seperate lines */
		if (opcodesize > 12) {
			i = 12;
			pagebreak(FALSE);
			phifg(DKBLUE,listfile);
			fprintf(listfile,"                      ");
			while (i < opcodesize) {
				fprintf(listfile,"%02X",pcbuf[i++]);
				if ((i %12) == 0) {
					fprintf(listfile,"\n");
					pagebreak(FALSE);
					phifg(DKBLUE,listfile);
					fprintf(listfile,"                      ");
				}
			}
			if (i%12 !=0)
				fputc('\n',listfile);
		}
	}
	fixlistnum = 0;
}
/* primary line list routine
 */
void dolist(void)
{
	if (macrolevel && macstyle != 2) {
		switch (macstyle) {
			case 1:
			case 3:
				if (!ifskip)
					dolist2();
				break;
			case 4:
				dolist2();
				break;
			case 5:
				if (opcodesize)
					dolist2();
				break;
		}
	}
	else {
				if (ifstyle == 2 || !ifskip)
					dolist2();
				}
}
/* display XREF info */
void printxref(BOOL dofile, char* filename,  EXPRESSION *p, int i)
{
	BOOL def = FALSE;
	int filenum;
	int linenum;
	long cur = p->xrefs[i];
	p->xrefs[i] = 0;
	if (cur < 0) {
		cur = -cur;
		def = TRUE;
	}
	if (dofile) {
		char *q;
		filenum = cur/1000000L;
		if (!filenum) {
			if ((q = strrchr(filename,'\\')) == 0)
				q = filename;
		}
		else {
			LIST *r;
			r = incfiles;
			for (i =1; i < filenum; i++)
				r = r->link;
			if ((q = strrchr(r->data,'\\')) == 0)
				q = r->data;
		}
		fprintf(listfile,"%27s%12s:",null,q);
	}
	else
		fputc(',',listfile);
	
	if (++xrefpos == 10) 
		fprintf(listfile,"\n%27s",null);

	linenum = cur %1000000L;
	fprintf(listfile,"%d",linenum);
	
	if (def)
		fputc('*',listfile);
}
/* list file tail (sections and symbols) */
void listtail(char *name)
{
	setequ = FALSE;
	if (prm_symdump && listfile) {
		int i, j, count=0,exppos=0;
		EXPRESSION **ptrtab = 0;
		LIST *p = sections;
	
		/* dump section data */
		pagebreak(TRUE);
		phifg(DKGREEN,listfile);
		fprintf(listfile,"Section Name\t\tLength\n");
		phifg(DKBLACK,listfile);
		while (p) {
			SECTIONREC *q = p->data;
			p = p->link;
			pagebreak(FALSE);
			fprintf(listfile,"%012s\t\t  %04X\n", q->name, q->highest);
		}
		pagebreak(FALSE);
		fputc('\n', listfile);
		pagebreak(FALSE);
		fputc('\n', listfile);
		
		/* count symbols */
  	for (i=0; i < HASH_TABLE_SIZE; i++) {
    	HASHREC *p;
    	if ( (p =hashtable[i]) != 0) {
				while (p) {
					if (!((EXPRESSION *)p)->isopcode)
						count++;
					p = p->link;
				}
			}
		}
		/* grab symbols */
		if (count) {
			ptrtab = AllocateMemory(count * sizeof(EXPRESSION));
			pagebreak(FALSE);		
  		for (i=0; i < HASH_TABLE_SIZE; i++) {
    		HASHREC *p;
    		if ( (p =hashtable[i]) != 0) {
					while (p) {
						if (!((EXPRESSION *)p)->isopcode)
							ptrtab[exppos++] = p;
						p = p->link;
					}
				}
			}
			/* sort symbols */
			for (i=0; i < count; i++)
				for (j=i+1; j < count; j++)
					if (phicmp(ptrtab[i]->name,ptrtab[j]->name) > 0)  {
						EXPRESSION *temp = ptrtab[i];
						ptrtab[i] = ptrtab[j];
						ptrtab[j] = temp;
					}
			/* dump symbols */
			pagebreak(FALSE);
			phifg(DKGREEN,listfile);
			fprintf(listfile,"                Symbol    Section     Type         Value           Size\n");
			phifg(DKORANGE,listfile);
			for (i=0; i < count; i++) {
				EXPRESSION *p = ptrtab[i];
				char symname[100];
				char *section = nosection;
				char *type=0;
				char flags[10];
				char size[10];
				long value = p->x.value;
				int symlen,len,i;
				int maybit = FALSE;
				symlen = len = strlen(p->name);
				if (phiused) {
					for (i=0; i < len; i++)
						if (p->name[i] & 0x80)
							symlen--;
				}
				for (i=0; i < 20 - symlen; i++)
					symname[i] = ' ';
				symname[i] = 0;
				strcat(symname, p->name);
				flags[0] = 0;
				if (p->section) {
					SECTIONREC *q = getsection(p->section);
					section = q->name;
				}
				if (!p->isdef)
					strcpy(flags,"U");
				if (p->isextern) {
					char *ptr = flags + strlen(flags);
					*ptr++ = mextern[(p->size)*2];
					*ptr++ = mextern[(p->size)*2+1];
					*ptr++ = 0;
				}
				if (p->size < 4) {
					size[0] = mextern[(p->size)*2+1];
					size[1] = 0;
				}
				else
					sprintf(size,"%06X",p->size);
				if (p->islabel)
					type = mlabel;
				if (p->ismacro) {
					type = mmacro;
					value = 0;
				}
				if (p->isoperand) {
					type = moperand;
					value = 0;
				}
				if (p->isstruct) {
					type = mstruct;
					value = 0;
				}
				if (!type) {
					if (p->ischangeable)
						type = mset;
					else 
						maybit = TRUE;
						if (p->type)
							type = typelist[(p->type & 0xff)-1];
						else
							type = mequ;
				}
				pagebreak(FALSE);
				if (maybit && lconfig & MODE8051 && p->type == TYPE8051BIT && value < 128) {
					fprintf(listfile,"  %s%011s%010s%05s%08lX (%02lX.%ld)  %s\n", symname, section, type,
								flags, value, 0x20 + (value>>3),value&7, size);

				}
				else fprintf(listfile,"  %s%011s%010s%05s%08lX        %s\n", symname, section, type,
								flags, value, size);
				/* dump xref info for symbol */
				if (prm_xref) {
					BOOL found;
					do {
						int i,val;
						found = FALSE;
						for (i=0; i < p->xrefcount; i++) {
							if (!found) {
								if (p->xrefs[i]) {
									xrefpos = 1;
									val = ABS(p->xrefs[i]/1000000L);
									found = TRUE;
									printxref(TRUE,name,p,i);
								}
							}
							else {
								if (p->xrefs[i]) {
									if (p->xrefs[i]/1000000L == val) {
										printxref(FALSE,name,p,i);
									}
								}
							}
						}
						if (found)
							fputc('\n',listfile);
					} while (found);
				}
			}
			DeallocateMemory(ptrtab);
		}
		
	}
	/* cleanup and init for next pass */
	listenabled = pagenabled = TRUE;
	xrefon(TRUE);
	pagenum = 1;
	pageline = PAGEMAX * 2;
}