/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * linker.c
 *
 * output the .o file
 */
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <dos.h>
#include "utype.h"
#include "umem.h"
#include "asm.h"
#include "output.h"
#include "sections.h"
#include "linker.h"

extern char *sectiontypes[];
extern BOOL prm_autodepends;
extern long basealign;
extern int usedrevision;
extern LIST *sections;
extern HASHREC **hashtable;
extern BOOL prm_debug;
extern int bytesize,wordsize;
extern char endian;
extern char progname[];
extern LIST *incfiles;
extern long lconfig;
extern char filesused[];
extern EXPRESSION *entrypoint;
BYTE checksum;
BOOL usefp = FALSE;
BOOL usedfloating = FALSE;

#define LINKASCII

#ifdef LINKASCII
/* FPRINTF with checksum */
static void putascii(FILE *out,char *format, ...)
{
	char buffer[256];
	int i,l;
	va_list ap;

	va_start(ap, format);
	vsprintf(buffer, format, ap);
	l = strlen(buffer);
	for (i=0; i < l; i++)
		if ((buffer[i] & 0xff) > 31)
			checksum+=buffer[i];
	va_end(ap);
	fprintf(out,buffer);
}

#endif
/*
 * put out a checksum record */
static void putcs(FILE *out, BOOL toclear)
{
	if (toclear) {
#ifdef LINKASCII
		putascii(out,"CS.\r\n");
#endif /* LINKASCII */
	}
	else {
#ifdef LINKASCII
		checksum += 'C';
		checksum += 'S';
		putascii(out,"CS%02X.\r\n",checksum & 127);
#endif
	}
	checksum = 0;
}
/* this is the file header */
static void link_header(FILE *out, char *name)
{
	struct date thedate;
	struct time thetime;
	char buf[20];
	char fpspec = 'I';  /* Default is integer type */
	int i = 1;
	/* processor name */
	getLinkName(usedrevision,buf);
	getdate(&thedate);
	gettime(&thetime);
	if (usedfloating)
		fpspec = 'F';
#ifdef LINKASCII
	/* put out linker init records */
	putascii(out,"MB%c%sPHIDOS,%02X%s.\r\n",fpspec,buf, strlen(name), name);
	putascii(out,"AD%d,%d,%c.\r\n",bytesize,wordsize,endian);
	/* put out the lconfig record */
	if (lconfig) 
		putascii(out,"CO104,08%08lX.\r\n",lconfig);
	/* put out the date & time */
	putascii(out,"DT%04d%02d%02d%02d%02d%02d.\r\n", 
			thedate.da_year,thedate.da_mon,thedate.da_day,
			thetime.ti_hour, thetime.ti_min, thetime.ti_sec);
#endif
	/* dump include files and autodepend information */
	while (incfiles) {
		LIST *p = incfiles->link;
#ifdef LINKASCII
		if (prm_autodepends)
			putascii(out,"CO102,%02X%s.\r\n",strlen(incfiles->data),incfiles->data);
		if (prm_debug && filesused[i++])
			putascii(out,"CO105,%02X%s.\r\n",strlen(incfiles->data),incfiles->data);
#endif
		DeallocateMemory(incfiles->data);
		DeallocateMemory(incfiles);
		incfiles = p;
	}
/* mark the end of the header */
#ifdef LINKASCII
	putascii(out,"CO101,07ENDHEAD.\r\n");
#endif /* LINKASCII */
	putcs(out, FALSE);
}

/* Trailer records */
static void link_trailer(FILE *out)
{
#ifdef LINKASCII
	/* Module end */
	putascii(out,"ME.\r\n");
#endif /* LINKASCII */
}
/* dump symbols */
static void putsyms(FILE *out)
{
  int i;
	int locid = 1;

	/* traverse hash table */
  for (i=0; i < HASH_TABLE_SIZE; i++) {
    EXPRESSION *p;
    if ( (p =hashtable[i]) != 0) {
			while (p) {
#ifdef LINKASCII
				if (p->ispublic) {
					/* public name and value */
					putascii(out,"NI%X,%02X%s.\r\n",p->id, strlen(p->name), p->name);
					if (!p->section)
						putascii(out,"ASI%X,%lX.\r\n", p->id, p->x.value);
					else
						putascii(out,"ASI%X,R%X,%lX,+.\r\n", p->id, p->section,p->x.value);
					if (p->type & 0x100)
						putascii(out,"ATI%X,%X.\r\n", p->id, p->type & 0xff);
				}
				else
					if (p->isextern) {
							/* extern name */
							putascii(out,"NX%X,%02X%s.\r\n",p->id, strlen(p->name), p->name);
					}
					else
						if (p->islabel && prm_debug){
							/* local symbols */
							putascii(out,"NN%X,%02X%s.\r\n",locid, strlen(p->name), p->name);
							putascii(out,"ASN%X,R%X,%X,+.\r\n", locid++, p->section,p->x.value);
						}
#endif /* LINKASCII */
				p = p->link;
			}
		}
	}  
	putcs(out, FALSE);
}
/* dump sections and program start */
static void putsections(FILE *file)
{
  LIST *q = sections;
	int i = 1;
	while (q) {
		SECTIONREC *p = q->data;
		char *typeattribs = sectiontypes[p->type & 0xff];
		if (p->highest) {
#ifdef LINKASCII
			/* section number, alignment, and size */
			putascii(file,"ST%X,C,%s%02X%s.\r\n",i,typeattribs,strlen(p->name), p->name);
			putascii(file,"SA%X,%lX.\r\n",i,basealign);
			putascii(file,"ASS%X,%X.\r\n",i, p->highest);
#endif /* LINKASCII */
		}
		q = q->link;
		i++;
	}
	/* entry point */
	if (entrypoint) 
#ifdef LINKASCII
			putascii(file,"ASG,R%X,%X,+.\r\n", entrypoint->section,entrypoint->x.value);
#endif /* LINKASCII		 */
	putcs(file, FALSE);
}
/* 
 * put a line of text (no fixups) */
static long putld(FILE *file, long start, long end, BYTE *buf)
{
	if (start == end)
		return(start);
	/* fill up multiple lines */
	while (end - start >= LDPERLINE) {
		int i = 0;
#ifdef LINKASCII
		putascii(file,"LD");
#endif
		for (i=0; i < LDPERLINE; i++) {
#ifdef LINKASCII
			putascii(file,"%02X",buf[start++]);
#endif /* LINKASCII */
		}
#ifdef LINKASCII
		putascii(file,".\r\n");
#endif /* LINKASCII */
	}
	/* if no partial lines */
	if (start == end)
		return(start);
#ifdef LINKASCII
	/* make a partial line */
	putascii(file,"LD");
#endif
	while (start < end) {
#ifdef LINKASCII
			putascii(file,"%02X",buf[start++]);
#endif /* LINKASCII */
	}
#ifdef LINKASCII
	putascii(file,".\r\n");
#endif /* LINKASCII */
	return(start);
}
/* gen a fixup */
void dolr(char *buf,FIXUP *p)
{
	char type;
	char buf1[20];
	int sectorid;
  buf1[0] = 0;
	/* throw in a math op */
	if (p->type & (FIXMINUS | FIXPLUS)) {
		dolr(buf1+1,p->link);
		buf1[0] = ',';
		if (p->type & FIXMINUS)
			strcat(buf1,",-");
		else
			strcat(buf1,",+");
	}
	sectorid = p->id;
	/* pick the vaiiable type to fix up */
	switch (p->type & FIXMASK) {
		case FIXEXTERN:
			type = 'X';
			break;
		case FIXPUBLIC:
			type = 'I';
			break;
		case FIXLOCALPUB:
			type = 'N';
			break;
		case FIXRELSECT:
			type = 'R';
			sectorid = p->section;
			break;
		case FIXVALUE:
		case FIXPC:
			break;
		default:
			Error("Internal error 2");
			break;
	}
	/* now gen the fixup based on the addressing mode used */
	if ((p->type & FIXMASK) == FIXPC)
		sprintf(buf,"P,%lX,-",p->reladdress);
	else {
		if ((p->type & FIXMASK) == FIXVALUE) {
		  if (p->relpc)
				sprintf(buf,"%lX,P,-,2,+",p->reladdress);
			else
				if (p->rel)
					sprintf(buf,"%lX,P,-",p->reladdress);
				else
					sprintf(buf,"%lX", p->reladdress);
		}
		else {
			if ((p->type & FIXMASK) == FIXRELSECT) {
		  	if (p->relpc)
					sprintf(buf,"R%X,%lX,+,P,-,2,+",sectorid,p->reladdress);
				else
					if (p->rel)
						sprintf(buf,"R%X,%lX,+,P,-",sectorid, p->reladdress);
					else
						sprintf(buf,"R%X,%lX,+", sectorid, p->reladdress);
			}
			else {
	  		if (p->relpc)
 					sprintf(buf,"%c%X,P,-,2,+",type,sectorid);
				else
					if (p->rel)
						sprintf(buf,"%c%X,P,-",type,sectorid);
					else
						sprintf(buf,"%c%X", type, sectorid);
			}
		}
	}
	strcat(buf,buf1);
}
/* main fixup routine */
static long putlr(FILE *file, FIXUP *p)
{
#ifdef LINKASCII
	char buf[40];
	long size=p->size, size1=size;
	int size2;
	if (size > 0xffff)
		size >>=16;
	switch (size) {
		case TBYTE:
		case TEVEN:
		case TINPAGE:
			size2 = 1;
			break;
		case TA11BJMP:
		case TA11BCALL:
		case TCOP8JSR:
		case TCOP8JMP:
		case TWORD:
			size2 = 2;
			break;
		case TLONG:
			if (size1 > 0xffff)
				size1+=0x10000L;
			else size1++;
			size2 = 4;
			break;
	}
	/* put the fixup */
	dolr(buf,p);
	putascii(file,"LR(%s,%lX).\r\n", buf, size1);

	return(size2);
#endif /* LINKASCII */
}
/* Dump all segment data (text and fixups )*/
static void putdata(FILE *file)
{
  LIST *q = sections;
	int i = 1;
	while (q) {
		SECTIONREC *p = q->data;
		long org = 0;
		if (p->highest) {
			LIST *l = p->fixuplist;
			putascii(file, "SB%X.\r\n",i);
			while (l) {
				FIXUP *f = l->data;
				org = putld(file, org, f->address, p->buffer);
				org += putlr(file, f);
				l = l->link;
			}
			putld(file, org, p->highest, p->buffer);
			putcs(file,FALSE);
		}
		i++;
		q = q->link;
	}
}

/* global linker routine, dumps the .o file */
void linkerout(FILE *file, char*name)
{
	checksum = 0;
	link_header(file, name);
	putsections(file);
	putsyms(file);
	putascii(file,"CO100,06ENDSYM.\r\n");
	putdata(file);
	link_trailer(file);
}