/*
 * Hash tables, conflicts are resolved by making each entry in the table
 * a linked list; if multiple sybols use the same hash they are just linked
 * together
 */

#include <string.h>
#include <stdio.h>
#include <memory.h>
#include "utype.h"
#include "hash.h"
#include "allocate.h"


/*
 * Compute hash for use in internal hash tables.
 * Librarian uses a different hash algorithm and a different table
 * structure, see libs.c for details
 */
static uint ComputeHash(char *string)
{
  uint len = strlen(string), rv;
  char *pe = len + string;
  const BYTE blank = ' ';

  rv = len | blank;
  while(len--) {
    BYTE cback = (*--pe) | blank;
    rv = ROTR(rv,2) ^ cback;
  }
  return(rv % HASH_TABLE_SIZE);
}
/*
 * Add an entry to the hash table.  Return previous entry if it exists
 */
HASHREC *AddHash(HASHREC **table, HASHREC *item, char *key)
{
  int index = ComputeHash(key);
  HASHREC **p;
  item->key = key;
  item->link = 0;

  if (*(p = &table[index])) {
    HASHREC *q = *p,*r = *p;
    while (q) {
			r = q;
      if (!strcmp(r->key,key))
				return(r);
			q = q->link;
		}
		r->link = item;
  }
  else
    *p = item;
  return(0);
}
/*
 * Find something in the hash table
 */
HASHREC **LookupHash(HASHREC **table, char *key)
{
  uint index = ComputeHash(key);
  HASHREC **p;

  if (*(p = &table[index])) {
    HASHREC *q= *p;
    while (q) {
      if (!strcmp(q->key, key))
				return(p);
			p = *p;
			q=q->link;
		}
	}
	return(0);
}
/*
 * Create a hash table
 */
HASHREC **CreateHashTable(void)
{
  HASHREC ** rv = (HASHREC **) AllocateMemory(HASH_TABLE_SIZE * sizeof(HASHREC *));
  memset(rv,0,HASH_TABLE_SIZE * sizeof(HASHREC *));
  return(rv);
}
/* 
 * Delete a hash table
 */
void RemoveHashTable(HASHREC ** t)
{
  DeallocateMemory(t);
}
#ifdef DEBUG
void DumpHashTable(HASHREC **t, char *string)
{
  int i;
  printf("%s\n\n",string);

  for (i=0; i < HASH_TABLE_SIZE; i++) {
    HASHREC *p;
    if ( (p =t[i]) != 0) {
			printf("index %d:\n",i);
			while (p) {
				printf("  Key: %s\n",p->key);
				p = p->link;
			}
		}
	}  
}
#endif DEBUG