/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * input.c
 *
 * input handler
 */
#include <stdio.h>       
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <memory.h>
#include "cmdline.h"
#include "umem.h"
#include "asm.h"
#include "input.h"
#include "gencode.h"
#include "macros.h"
#include "sections.h"
#include "loclab.h"
#include "olist.h"
#include "expr.h"
#include "interp.h"
#include "fpmath.h"
#include "access.p"

extern int basetype;
extern FILE *listfile;
extern int pass;
extern int opcodesize;
extern LIST *undeflist;
extern long org;
extern int macrolevel;
extern int sectionnum;
extern char *prm_includepath;
extern int ifskip;
extern BOOL prm_case_sensitive;
extern PHITEXT macrophiparms, phiparms;
extern int currentphifg;
extern BOOL putback;
extern int repeat;
extern BOOL phiused;
extern BOOL prm_autodepends;
extern BOOL prm_xref;
extern BOOL onxref;
extern int prm_errcount;

HASHREC **hashtable=0;
char buf[BUFLEN];
char *bufptr;
char *macroptr;
BOOL firstsymbol;
int level;
int lineno[INCLUDELEVELS];
FILE *ifile[INCLUDELEVELS];
char *filenames[INCLUDELEVELS];
int errcount;
BOOL newline;
BOOL loadsdata;
BOOL okxref = TRUE;

LIST *incfiles = 0;
extern OPCODEDATA dispatchtable[];

int curfile;

void BadTick(void)
{
	Error("Invalid expression");
}
/* set a new xref level */
void xrefon(unsigned v)
{
	okxref = v;
}
		
/* open an input/include file */
void openFile(char *string, char *mode)
{
	char buffer[256];
	strcpy(buffer,string);
	if (level == INCLUDELEVELS)
		fatal("Too many levels of include files\n");
	level++;
	lineno[level] = 0;
	filenames[level]= AllocateMemory(strlen(string)+1);
  strcpy(filenames[level],string);
	if (!(ifile[level] = SearchPath(buffer, prm_includepath, mode))) {
		fatal("Can't open %s\n",string);
	}
  newline = TRUE;
}
/* close a file */
void closeFile(void)
{
	LIST *p = &incfiles,*q;
	int i = 0;
	if (level == 1)
		curfile = 0;
	else
		while (p) {
			q = p->link;
			curfile = i++;
			if (!strcmp(p->data,ifile[level-1]) || !q)
		  	break;
			p = q;
		}
	fclose(ifile[level]);
	DeallocateMemory(filenames[level]);
	level--;
}
/* include nesting */
void setinclude(char *string)
{
	char buf[100];
	LIST *p = &incfiles,*q;
	int i = curfile = 1;
	if (ifskip)
		return;
	for (i=0; i <strlen(string); i++)
		buf[i] = toupper(string[i]);
	buf[i] = 0;
	openFile(buf,"r");
	i = 0;
	while (p) {
		q = p->link;
		curfile = i++;
		if (!strcmp(p->data,buf))
			return;
		if (!q)
			break;
		p = q;
	}
	q = AllocateMemory(sizeof(LIST));
	q->data = AllocateMemory(strlen(string)+1);
	strcpy(q->data,buf);
	q->link=0;
	p->link = q;
}
/* read the next line */
BOOL nextLine(char *buf, int len)
{
	/* check if in macro */
	if (macrolevel)
		if (GetMacroLine())
			return(TRUE);
		else
			phiparms = macrophiparms;
	/* close files until we hit one that isn't empty */
	if (level < 0 || feof(ifile[level]))
		return(FALSE);
	while (!feof(ifile[level])) {
		philine(buf,len,ifile[level]);
		lineno[level]++;
		if (buf[0] != 0) {
			return(TRUE);
		}
	}
	return(FALSE);
}
/* skip over spaces */
int skipSpace(char **buf)
{
	int curchar;
	while (iswhitespacechar(curchar = parsechar(buf))) ;
	return(curchar);
}
/*
 * parser errors */
void yyerror(char *msg)
{
	Error(msg);
	opcodesize = 0;
	dolist();
}
/* Error handler */
void Error(char *msg)
{
	if (pass == 2 && level >= 0) {
		if (errcount < prm_errcount)
  		printf("%s(%d): %s\n",filenames[level],lineno[level],msg);	
		else {
			if (listfile) {
				int oldforeground = currentphifg;
				pagebreak(FALSE);
				phifg(DKRED,listfile);
		  	fprintf(listfile,"%s(%d): %s\n",filenames[level],lineno[level],msg);
				phifg(oldforeground,listfile);
			}
		}
	  errcount++;
	}
}
/* fail keyword */
void fail(EXPRESSION *exp)
{
	char buf[100];
	if (ifskip)
		return;
	if (numericExpression(exp) && exp->isdef && !exp->isextern && !exp->relmode) {
		sprintf(buf, "Failure #%d", exp->x.value);
	}
	else
		sprintf(buf,"bad FAIL statement");
	Error(buf);
	dolist();
}
/* Msg keyword */
void putmsg(char *string)
{
	if (pass == 2) {
		printf("%s\n", string);
		if (listfile) {
			pagebreak(FALSE);
	  	fprintf(listfile,"%s\n", string);
		}
	}
}
/* Mark xref if defined */
void definedxref(EXPRESSION *p)
{
	if (prm_xref && okxref && pass == 1 && p->xrefs[p->xrefcount-1] > 0)
		p->xrefs[p->xrefcount-1] = - p->xrefs[p->xrefcount-1];
}
/* Add a new xref to the array */
void addxref(EXPRESSION *p, BOOL defd)
{
	if (!prm_xref || !okxref || pass != 1)
		return;
	if (p->xrefcount == 0)
		p->xrefs = AllocateMemory(sizeof(long)*5);
	else
		if (p->xrefcount %5 == 0)
			p->xrefs = ReallocateMemory(p->xrefs,sizeof(long) * (p->xrefcount+5));
	p->xrefs[p->xrefcount] = lineno[level];
  if (level)
		p->xrefs[p->xrefcount] += 1000000L * (curfile+1);
	p->xrefcount++;
	if (defd)
		definedxref(p);
}
/* Symbol table add */	
BOOL AddToTable(EXPRESSION *p)
{
  EXPRESSION *q;
	if (ifskip) {
		p->isintermed = TRUE;
		delexpr(p,0);
		return(FALSE);
	}
	/* check for local labels */
	if (p->islabel) {
		if (registerlocallabel(p))
			return(FALSE);
		p->section = sectionnum;
	}
	else
		p->section = 0;
	p->defpass = pass;
	q = AddHash(hashtable,p);
	if (q) {
		/* Redefined, either use error or update old definition */
		if (q->defpass == pass) {
			if ((q->ispublic || q->isextern || q->relmode || q->islocalpub) && !q->isdef) {
				if (q->isextern)
					Error("Defined XREF not allowed");
				else {
					if (q->relmode)
						Error("Can't use external addition here");
					else {
						if (!sectionnum)
							Error("Can't define publics outside of section");
						addxref(q,TRUE);
						definedxref(q);
						q->type = basetype;
						q->isdef = p->isdef;
						q->section = p->section;
						q->x.value = p->x.value;
						q->size = p->size;
						q->islabel = p->islabel;
						registerlocallabel(q);
						if (p->ismacro || p->isoperand || p-> ischangeable || p->ismovemreg || p->isstruct || !p->islabel)
							Error("Invalid XDEF type");
					}
				}
			}
			else {
				char buf[BUFLEN];
				sprintf(buf,"%s redefined",p->name);
				Error(buf);
			}
		}
		q->defpass = pass;
		return(TRUE);
	}
	else
			addxref(p,TRUE);
	return(FALSE);
}

/* Load the instruction set into the symbol table */
void loadtable(OPCODEDATA *tbl)
{
	int i = 0;
	while (tbl[i].text) {
		EXPRESSION *p  = AllocateMemory(sizeof(EXPRESSION));
		p->name = tbl[i].text;
		p->reg1 = p->reg2 = 0;
		p->size = 0;
		p->link = 0;
		p->loclablink = 0;
		p->isdef = FALSE;
		p->isintermed = FALSE;
		p->islabel = FALSE;
		p->isextern = FALSE;
		p->defpass = 3;
		p->isopcode = TRUE;
		p->ismacro = FALSE;
		p->isoperand = FALSE;
		p->ischangeable = FALSE;
		p->ismovemreg = FALSE;
		p->isstruct = FALSE;
		p->rel = FALSE;
		p->ispublic = FALSE;
		p->islocalpub = FALSE;
		p->xrefcount = 0;
	  p->swapper = 0xffff;
		p->relpc = FALSE;
		p->relmode = RM_NONE;
		p->section = 0;
		p->isfp = FALSE;
		p->type = basetype;
		p->floatspec = FALSE;
		p->id = 0;
			p->x.xdata = &tbl[i];
		AddHash(hashtable, p);
		i++;
	}
}
void input_init(FILE *filename)
{
	firstsymbol = FALSE;
	bufptr = buf;
	level = -1;
	newline = FALSE;
	loadsdata = FALSE;
	buf[0] = 0;
	curfile = 0;
	openFile(filename,"r");
	if (!hashtable) {
  	hashtable = CreateHashTable(HASH_TABLE_SIZE);
		loadtable(dispatchtable);
	}
}

	
	