/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * expr.c
 *
 * expression evaluation
 */
#include <stdio.h>                     
#include <memory.h>
#include <string.h>
#include "utype.h"
#include "umem.h"
#include "asm.h"
#include "expr.h"
#include "gencode.h"
#include "interp.h"

extern int basetype;
extern int pass;
extern BOOL setequ;
extern BYTE setequval[12];
extern int sectionnum;
extern BOOL setequ;
extern HASHREC **hashtable;
extern BOOL ifskip;
extern HASHREC **hashtable;

BOOL setequfloat;

/* error when label expected */
void labelexpected(void)
{
  Error("Label Expected");
}
/* make a bit mask */
unsigned makebits(int start, int end, int ofs)
{
  unsigned rv=0;
  int i;
	if (start < 8 && end < 8) {
    unsigned mask = 1 << (ofs+start);
  	for (i=start;i<=end;i++) {
			rv |= mask;
			mask<<=1;
		}
	}
	else
		illop();
	return(rv);
}
  
/* routine to create a new expression for the result */
EXPRESSION *makexpr(long value, int isdef, char *name, int isintermed)
{
  EXPRESSION *rv = AllocateMemory(sizeof(EXPRESSION));
  rv->x.value = value;
  rv->isdef = isdef;
  if (name) {
		rv->name = AllocateMemory(strlen(name)+1);
		strcpy(rv->name, name);
	}
	else
		rv->name = 0;
	rv->reg1 = rv->reg2 = 0;
	rv->loclablink = 0;
	rv->islabel = FALSE;
  rv->isintermed = isintermed;
	rv->size = 0;
	rv->isextern = 0;
	rv->defpass = 0;
  rv->isopcode = FALSE;
  rv->ismacro = FALSE;
  rv->isoperand = FALSE;
  rv->ismovemreg = FALSE;
  rv->ischangeable = FALSE;
	rv->isstruct = FALSE;
	rv->rel = FALSE;
	rv->ispublic = FALSE;
	rv->islocalpub = FALSE;
	rv->xrefcount = 0;
	rv->swapper = 0xffff;
	rv->relpc = FALSE;
	rv->relmode = RM_NONE;
	rv->id = 0;
	rv->section = 0;
	rv->isfp = 0;
	rv->type = basetype;
	rv->floatspec = 0;
	rv->type = 0;
	return(rv);
}
EXPRESSION *maketypedexpression(long value, int isdef, char *name, int isintermed, int type)
{
	EXPRESSION *rv = makexpr(value,isdef,name,isintermed);
	rv->type = type;
}
/*
 * Make a new expression, setting the DEFINED flag appropriately */
EXPRESSION *defexpr(long value, EXPRESSION *exp1, EXPRESSION *exp2)
{
	EXPRESSION *rv = exp1;
	if (!numericExpression(exp1) || (exp2 && !numericExpression(exp2))) {
		delexpr(exp1,exp2);
		rv = makexpr(0,FALSE, 0, TRUE);
		return(rv);            
	}
	if (exp2)  {
		rv = (makexpr(value, exp1->isdef && exp2->isdef, 0, TRUE));
		delexpr(exp1,exp2);
	}
	else {
		delexpr(exp2, 0);
		exp1->x.value = value;
	}
	return(rv);
}
/* See if two numbers are equal */
EXPRESSION *eqexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value == exp2->x.value,exp1,exp2);
	return(rv);
}
/* test for inequality */
EXPRESSION *neexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value != exp2->x.value,exp1,exp2);
	return(rv);
}
/* Test for greater than */
EXPRESSION *gtexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value > exp2->x.value,exp1,exp2);
	return(rv);
}
/* test for less than */
EXPRESSION *ltexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value < exp2->x.value,exp1,exp2);
	return(rv);
}
/* test for greator-or-equal */
EXPRESSION *geexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value >= exp2->x.value,exp1,exp2);
	return(rv);
}
/* test for less-or-equal */
EXPRESSION *leexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value <= exp2->x.value,exp1,exp2);
	return(rv);
}
/* AND two expressions */
EXPRESSION *andexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value & exp2->x.value,exp1,exp2);
	return(rv);
}
/* Or two expressions */
EXPRESSION *orexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv  = defexpr(exp1->x.value | exp2->x.value,exp1,exp2);
	return(rv);
}
/* Logical or two expressions */
EXPRESSION *logor(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value || exp2->x.value,exp1,exp2);
	return(rv);
}
/* logical and two expressions */
EXPRESSION *logand(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv  = defexpr(exp1->x.value && exp2->x.value,exp1,exp2);
	return(rv);
}
/* xor two expressions */
EXPRESSION *xorexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value ^ exp2->x.value,exp1,exp2);
	return(rv);
}
/* add two expressions */
EXPRESSION *addexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	if (exp1->relmode || exp2->relmode)
		Error("Can't use multiple rel additions");
	if (exp1->isextern) {
		exp1->relmode = RM_PEXT;
		exp1->ext = exp2;
		return(exp1);
	}
	else
		if (exp2->isextern) {
			exp2->relmode = RM_PEXT;
			exp2->ext = exp1;
			return(exp2);
		}
		else
			if (exp1->section != exp2->section)
				if (exp1->section && LookupHash(hashtable,exp1->name)) {
					exp1->relmode = RM_PPUB;
					exp1->ext = exp2;
					return(exp1);
				}
				else
					if (exp2->section && LookupHash(hashtable,exp2->name)) {
						exp2->relmode = RM_PPUB;
						exp2->ext = exp1;
						return(exp2);
					}
	rv = defexpr(exp1->x.value + exp2->x.value,exp1,exp2);
	return(rv);
}
/* subtract two expressions */
EXPRESSION *subexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	if (exp1->relmode || exp2->relmode)
		Error("Can't use multiple relative additions");
	if (exp1->isextern) {
		exp1->relmode = RM_MEXT;
		exp1->ext = exp2;
		return(exp1);
	}
	else
		if (exp2->isextern) {
			exp1->relmode = RM_MEXT;
			exp1->ext = exp2;
			return(exp1);
		}
		else
			if (exp1->section && (exp1->section != exp2->section) && LookupHash(hashtable,exp1->name)) {
				exp1->relmode = RM_MPUB;
				exp1->ext = exp2;
				return(exp1);
			}
	rv = defexpr(exp1->x.value - exp2->x.value,exp1,exp2);
	return(rv);
}
/* multiply two expressions */
EXPRESSION *mulexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value * exp2->x.value,exp1,exp2);
	return(rv);
}
/* scale an expression */
EXPRESSION *scaleexp(EXPRESSION *exp1, long scale)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value <<scale,exp1,0);
	return(rv);
}
/* divide two expressions */
EXPRESSION *divexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value / exp2->x.value,exp1,exp2);
	return(rv);
}
/* Take the moduluos */
EXPRESSION *modexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value % exp2->x.value,exp1,exp2);
	return(rv);
}
/* shift left */
EXPRESSION *shlexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value << exp2->x.value,exp1,exp2);
	return(rv);
}
/* shift right */
EXPRESSION *shrexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value >> exp2->x.value,exp1,exp2);
	return(rv);
}
/* negate */
EXPRESSION *negexp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(-exp1->x.value,exp1,0);
	return(rv);
}
/* bitwise complement */
EXPRESSION *bitcomp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(~exp1->x.value,exp1,0);
	return(rv);
}
/* logical complement */
EXPRESSION *logcomp(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(!exp1->x.value,exp1,0);
	return(rv);
}
/* see if two strings are equal */
EXPRESSION *eqstring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = !phicmp(string1,string2);
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* strings not equal */
EXPRESSION *nestring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = phicmp(string1,string2);
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* string greater than */
EXPRESSION *gtstring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = phicmp(string1,string2) > 0;
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* string less than */
EXPRESSION *ltstring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = phicmp(string1,string2) < 0;
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* string greater or equal */
EXPRESSION *gestring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = phicmp(string1,string2) >= 0;
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* string less or equal */
EXPRESSION *lestring(char *string1, char *string2)
{
  EXPRESSION *rv;
  long val = phicmp(string1,string2) <= 0;
	rv = makexpr(val, TRUE, 0 , TRUE );
	return(rv);
}
/* seems to copy an expression... */
EXPRESSION *toabs(EXPRESSION *exp1, EXPRESSION *exp2)
{
  EXPRESSION *rv;
	rv = defexpr(exp1->x.value,exp1,0);
	return(rv);
}
/* handle swap for the MODE command */
unsigned long doswap(long spec, long old)
{
  unsigned long val = 0;
	int i;
	if (spec == 0xffff)
		return old;
	for (i=0; i< 4; i++) {
		int grab = spec & 0x0f;
		spec = spec >> 4;
		val = val >> 8;
		switch (grab) {
			case 0:
				val |= (old & 0xff) << 24;
				break;
			case 1:
				val |= (old & 0xff00) <<16;
				break;
			case 2:
				val |= (old & 0xff0000L) << 8;
				break;
			case 3:
				val |= (old & 0xff000000L);
				break;
		}
	}
	return(val);
}
/* create a swapper */
EXPRESSION *swapper(long spec, EXPRESSION *exp)
{
	exp->swapper = spec;
	return exp;
}
/* get the size of an expression */
EXPRESSION *sizeofexpr(EXPRESSION *exp)
{
	int val;
	EXPRESSION *oldexp = exp;
	EXPRESSION **q = LookupHash(hashtable, exp->name);
	if (!q)
		return(exp);
	exp = *q;
	if (exp->isstruct)
		val = exp->size;
	else {
		if (!exp->isdef)
			val = 4;
		else
			switch(exp->size) {
				case TBYTE:
					val = 1;
					break;
				case TWORD:
					val = 2;
					break;
				case TLONG:
				case 0 :
					val = 4;
					break;
				default:
					val = exp->size;
			}
	}
	return(defexpr(val,oldexp,0));
}
norelequate(EXPRESSION *q, EXPRESSION *exp)
{
	if (exp->islabel)  {
		Error("Can't equate to a relative value");
		q->x.value = 0;
	}
}
/* EQU */
EXPRESSION *equexp(EXPRESSION *label, EXPRESSION *exp)
{
	EXPRESSION *q = 0;
	if (!ifskip) {
		if (label == 0) 
			labelexpected();
		else 
			if (!numericExpression(exp) || !exp->isdef)
				badexpression();
			else {
				label->isfp = FALSE;
				label->islabel = FALSE;
				label->x.value = exp->x.value;
				label->isintermed = FALSE;
				label->section = 0;
	 		q = AddHash(hashtable,label);
			if (q) {
					norelequate(q,exp);
					q->islabel = FALSE;
					q->ischangeable = FALSE;
					q->x.value = label->x.value;
					q->isfp = FALSE;
					q->section = 0;
				q->isintermed = TRUE;
			}
				*((long*)setequval) = exp->x.value;
				setequ = TRUE;	
			}
	}
	delexpr(exp,0);
	dolist();
	return q;
}
void equtypedexp(EXPRESSION *label, EXPRESSION *exp, int type)
{
	EXPRESSION *rv =equexp(label,exp);
	if (rv)
		rv->type = type;
	return rv;
}
			
/* SET OR = */
void setexp(EXPRESSION *label, EXPRESSION *exp)
{
	if (!ifskip) {
	  EXPRESSION *q;
		if (!numericExpression(exp))
			badexpression();
		else {
			label->isfp = FALSE;
			label->ischangeable = TRUE;
			label->x.value = exp->x.value;
			label->isintermed = FALSE;
			label->section = 0;
			label->defpass = pass;
	
	 		q = AddHash(hashtable,label);
			if (q) {
				if (!q->ischangeable)
					Error("Symbol can't be redefined");
				else {
					norelequate(q,exp);
					q->x.value = label->x.value;
					q->isfp = FALSE;
					q->defpass = label->defpass;
				}
				label->isintermed = TRUE;
			}
			*((long *)setequval) = label->x.value;
			setequ = TRUE;
			setequfloat = FALSE;
		}
	}
	else
		label->isintermed = TRUE;
	delexpr(exp, label);
	dolist();
}