/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * asm.c
 *
 *main program code
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "cmdline.h"
#include "umem.h"
#include "asm.h"
#include "sections.h"
#include "gencode.h"
#include "input.h"
#include "linker.h"
#include "output.h"
#include "interp.h"
#include "expr.h"

extern HASHREC **hashtable;
extern basealign;
extern long org;
extern int errcount;
extern BOOL setequ;
extern BOOL phiused;
extern long memory_left;
extern BOOL ToFixup;
extern BOOL setequ;
extern char srcname[];

char *curname;

/* Command line parameters */
BOOL prm_xref = FALSE;
BOOL prm_autodepends = FALSE;
BOOL prm_case_sensitive = TRUE;		/* case sensitive flag */
uint prm_errcount = 25;							/* Max errors to collect */
char *prm_includepath = 0;						/* Search path for libraries */
BOOL prm_list = FALSE;
extern uint prm_procrevision;			/* Processor type */
BOOL prm_symdump = TRUE;			/* Whether or not to do a symbol dump */
BOOL prm_binary = FALSE;		/* Whether or not to do binary output */
BOOL prm_debug = FALSE;
// Global varieables
LIST *asmlist = 0;								/* List of object files  */
char *ofile = 0;									/* EXE file name */
char *lisfile = 0;									/* MAP file name */
int pass = 1;
FILE *listfile = 0;
FILE *outfile  = 0;
LIST *deflist = 0;

char *usage_text = "[/a/b/l/s/u/v/x/E/I/V] file list";

void BoolSetup(char select, char *string);
void ErrorSetup(char select, char *string);
void IncludeSetup(char select, char *string);
void targetsetup(char select, char *string);
void DefineSetup(char select, char *string);

ARGLIST ArgList[] = {
	{ 'a', ARG_BOOL, BoolSetup },
	{ 'b', ARG_BOOL, BoolSetup },
	{ 'c', ARG_BOOL, BoolSetup },
  { 'l', ARG_BOOL, BoolSetup },
  { 's', ARG_BOOL, BoolSetup },
  { 'u', ARG_BOOL, BoolSetup },
  { 'x', ARG_BOOL, BoolSetup },
  { 'v', ARG_BOOL, BoolSetup },
	{ 'D', ARG_CONCATSTRING, DefineSetup },
  { 'E', ARG_CONCATSTRING, ErrorSetup },
  { 'I', ARG_CONCATSTRING, IncludeSetup },
  { 'V', ARG_CONCATSTRING, targetsetup },
  { 0, 0, 0 }
} ;
/*
 * Setup for boolean command line args
 */
static void BoolSetup(char select, char *string)
{
  switch(select) {
		case 'a':
			prm_autodepends = TRUE;
			break;
		case 'b':
			prm_binary = TRUE;
			break;
		case 'u':
			prm_case_sensitive = FALSE;
			break;
		case 'l':
			prm_list = TRUE;
			break;
		case 's':
			prm_symdump = FALSE;
			break;
		case 'v':
			prm_debug = TRUE;
			break;
		case 'x':
			prm_xref = TRUE;
			break;
  }
}
/* Setup for /D switch
 */
static void DefineSetup(char select, char *string)
{
	char *entry = malloc(strlen(string)+1);
	LIST *list = malloc(sizeof(LIST));
	strcpy(entry,string);
	list->link = deflist;
	deflist = list;
	list->data = entry;
}	
/*
 * Setup for the /E switch
 */
static void ErrorSetup(char select, char *string)
{
  prm_errcount = atoi(string);
}
/*
 * Setup for library search paths /L
 */
static void IncludeSetup(char select, char *string)
{
  uint len = strlen(string)+1;
  prm_includepath = (char *)malloc(len);
  strcpy(prm_includepath, string);
}
/*
 * Setup for target type
 */
static void targetsetup(char select, char *string)
{
	if (poption(string))
		printf("Invalid processor type specified, default used\n");
}
/*
 * Build Defines from command line
 */
void BuildDefines(void)
{
	LIST *list = deflist;
	long val;
	while (list) {
		char buf[100],*data2;
		EXPRESSION *p;
		strcpy(buf,list->data);
		if ((data2 = strchr(buf,'=')) != 0)
			*data2++ = 0;
		if (data2)
			val = atol(data2);
		else val = 0;
		if (!prm_case_sensitive) {
			int i;
			for (i=0; i <strlen(buf); i++)
				toupper(buf[i]);
		}
		p = makexpr(val,1,buf,0);
		p->defpass = 100;
		p->section = 0;
		AddToTable(p);
		list = list->link;
	}
}
/*
 * Insert a file onto one of the lists.  .LIB files go on library list,
 *   anything else is assumed an .obj file regardless of extension
 */
void InsertAnyFile(FILE *inf, FILE *outf, char *filename, char *path, int drive)
{
  char *newbuffer, buffer[100],*p = buffer;
	LIST *r = &asmlist;

	if (drive != -1) {
		*p++ = drive + 'A';
		*p++ = ':';
	}
	if (path) {
		strcpy(p,path);
		strcat(p,"\\");
	}
	else
		*p = 0;
  /* Allocate buffer and make .SRC if no extension */
	strcat(buffer,filename);
  AddExt(buffer,srcname);
  newbuffer = (char *) malloc(strlen(buffer) + 1);
  strcpy(newbuffer,buffer);

  /* Insert file */
	while (r->link)
		r = r->link;
	r->link = malloc(sizeof(LIST));
	r = r->link;
	r->link = 0;
	r->data = newbuffer;
}
/*
 * Main routine
 */
int asmmain(int argc, char *argv[])
{
  LIST *p;
	int rv = 0;
  char buffer[100];

  /* Scan command line for switches */
  if (!parse_args(&argc,argv,TRUE) || (argc == 1))
    usage(argv[0]);

	if (prm_xref || !prm_symdump)
		prm_list = TRUE;

  /* Scan the command line for file names or response files */
	FileRecurse(argc-1,&argv[1],0,InsertAnyFile,srcname,FALSE);

  /* If no response file, make up EXE and MAP file names from first .obj file */

	p = asmlist;
	if (!p) {
		printf("Nothing to do!\n");
		return(0);
	}
	while (p) {
		char *q;
		if ((q= strrchr(p->data,'\\')) != 0)
			q++;
		else
			if (( q = strchr(p->data,':')) != 0)
				q++;
			else
				q = p->data;

		curname = p->data;
			
    /* EXE file name */
		MemoryInit(0xffff);
    strcpy(buffer, q);
		StripExt(buffer);
		if (prm_binary)
			AddExt(buffer,".bin");
		else
			AddExt(buffer,".o");
	  ofile= AllocateMemory(strlen(buffer) +1);
    strcpy(ofile,buffer);

    /* MAP file name */
		StripExt(buffer);
    AddExt(buffer,".LIS");
	  lisfile= AllocateMemory(strlen(buffer) +1);
    strcpy(lisfile,buffer);

		setequ = FALSE;
		pass = 1;
		printf("File: %s\n",(char *)p->data);
		gencodeinit();
		dcinit();
		PhiInit();
		ReinitIf();
		input_init((char *)p->data);
  	loclabinit();
		extinit();
		section_init();
		strucini();
		macini();
		BuildDefines();
		floatinit();
  	yyparse();

		if (prm_list) {
			listfile = fopen(lisfile,"w");
			if (!listfile)
				fatal("Can't open list file %s",lisfile);
			if (phiused)
				fputc('\x1f',listfile);
		}
		else
			listfile = 0;
		setequ = FALSE;
		pass = 2;
		gencodeinit();
		dcinit();
		setsectionhighs();
  	loclabinit();
		ReinitIf();
		input_init((char *)p->data);
		strucini();
		errcount = 0;
		PhiInit();
		macini();
		ToFixup = TRUE;
		floatinit();
		yyparse();
		checksectionhighs();

  	listtail(p->data);

		if (listfile)
			fclose(listfile);

		if (errcount) {
			printf("\n%d Errors",errcount);
			if (errcount > prm_errcount)
				printf(" %d shown\n",prm_errcount);
			else
				putc('\n',stdout);
		}
		else {
			outfile = fopen(ofile,"wb");
			if (!outfile)
				fatal("Can't open output file %s",ofile);
			if (prm_binary)
				do_output(outfile);
			else
				linkerout(outfile, p->data);
			fclose(outfile);
		}
		gc_rundown();
		p = p->link;
		printf("   %ldK left.\n",memory_left/1024L);
		hashtable = 0;
		MemoryRundown();
		if (errcount) {
			rv = 1;
		}
	}		
  return(rv);
}