;
; LSD
;
; Copyright(c) LADsoft
;
; David Lindauer, camille@bluegrass.net
;
; Mouse.asm
;
; Function: Provide mouse and touch-panel support
;   handle MOUSE OS calls
;   handle parsing mouse input
;   handle drawing cursor in either text or VGA mode
;
;   assumes AGILER mouse
;   assumes mouse is turned off if graphics update is in progress
;   Update is done via the 18.2Hz timer interrupt so if the mouse is moved
;     during a graphics update and is on VGA regs will be overwritten and
;     not restored!
;
	;MASM MODE
	.386p

include  segs.asi 
include  os.asi 
include  serial.asi 
include  mouse.asi 
include  dispatch.ase 
include  prints.ase 
include  vgaini.ase 
include  video.ase 
include  vga.asi 
include  vga.mac 

AGILER = 1
FL_SHOWN = 1
FL_INUSE = 0

CURSORLINES = 16
CURSORCOLS = 3

	PUBLIC	AttachMouse, DetachMouse, MouseIntr, MouseHandler
	PUBLIC	HideMouse, ShowMouse
seg386data	SEGMENT	
dgrcurs	db	03fh, 0ffh, 01fh, 0ffh, 00fh, 0ffh, 007h, 0ffh
	db	003h, 0ffh, 001h, 0ffh, 000h, 0ffh, 000h, 07fh
	db	000h, 03fh, 000h, 01fh, 001h, 0ffh, 010h, 0ffh
	db	030h, 0ffh, 0f8h, 07fh, 0f8h, 07fh, 0fch, 07fh
	db	000h, 000h, 040h, 000h, 060h, 000h, 070h, 000h
	db	078h, 000h, 07ch, 000h, 07eh, 000h, 07fh, 000h
	db	07fh, 080h, 07ch, 000h, 06ch, 000h, 046h, 000h
	db	006h, 000h, 003h, 000h, 003h, 000h, 000h, 000h
dhotspot db	0,0
inuse	dd	0
port	dd	0
col	dd	0
row	dd	0
colinscale dd	0
coloutscale dd	0
rowinscale dd	0
rowoutscale dd	0
maxrow	dd	0
maxcol	dd	0
uhot	dd	0
lhot	dd	0
state	db	0
leftcount db	0
rightcount db	0
	align
overwritebuffer db 3*16*4 DUP (0)
grcursand	db	3*16 DUP (0)
grcursxor	db	3*16 DUP (0)
seg386data	ENDS	


seg386	SEGMENT	
;
; Attach the mouse
;
AttachMouse	PROC	
	call	DetachMouse		; Detach if exists
	os	SP_ENABLE		; Enable the serial port
	jc	noattach		; Get out if error
	push	ecx			;
	push	edx			;
	mov	[state],0		; State = 0
	mov	[leftcount],0		; Button Counts = 0
	mov	[rightcount],0		;
	mov	[port],ebx		; Port = EBX
	mov	[inuse],1 SHL FL_INUSE	; Mouse is inuse and hidden
	mov	[rowinscale],1		; Scale factors = 1
	mov	[rowoutscale],1		;
	mov	[colinscale],1		;
	mov	[coloutscale],1		;
	mov	[maxcol],639
	mov	[maxrow],479
	test	[theGraphicsMode],1
	jnz	short gotgraphicsmax
	mov	[maxrow],399
gotgraphicsmax:
	mov	eax,[maxcol]		; Adjust pos to middle
	shr	eax,1			;
	mov	[col],eax		;
	mov	eax,[maxrow]		;
	shr	eax,1			;
	mov	[row],eax		;
	mov	dx,WORD PTR [dhotspot]		; Get default hot spot
	call	LoadHotSpot
	os	SP_CLEARRCV		; Clear input buffer
	push	ds
	pop	fs
	mov	esi,offset dgroup:dgrcurs	; Get default graphics cursor
	call	CopyGraphicsCursor	; Copy to buffer

ifdef AGILER
	mov	ecx,1200		; 1200 Baud
	mov	edx,WORD7 OR PARITYNONE OR ((MCRRTS OR MCRDTR)SHL 8)
endif

	os	SP_SETMODE		; Set serial line mode
	pop	edx			;
	pop	ecx			;
noattach:
	ret
AttachMouse	ENDP	
;
; Detach the mouse
;
DetachMouse	PROC	
	bt	[inuse],FL_INUSE	; If not inuse get out
	jnc	short nodetach		;
	call	HideMouse		; Hide it
	mov	[inuse],0		; Not in use any more
	push	ebx			;
	mov	ebx,[port]		; Get mouse port
	os	SP_DISABLE		; Disable it
	pop	ebx			;
nodetach:
	ret
DetachMouse	ENDP	
;
; Mouse timer interrupt
;
MouseIntr	PROC	
	bt	[inuse],FL_INUSE	; In use?
	jnc	nointr			; No, get out
	push	ebx			; Push regs we wipe
	push	ecx			;
	push	edx			;
	push	esi			;
	push	edi
	push	es
	push	DS386
	pop	es
	bt	[inuse],FL_SHOWN	; See if shown
	jc	short update		; Continue if so
        mov	ebx,[port]		; Clear the buffer if not shown
	os	SP_CLEARRCV		;
	jmp	noredraw		; And get out
update:
	mov	ebx,[port]		; Get bytes in input buffer
	os	SP_RCVSTATUS		;
	mov	ecx,eax			; To ECX
	or	eax,eax
	jz	noredraw
	push	ecx
	call	HideMouse
	pop	ecx
	push	eax

ifdef AGILER
	cmp	ecx,3			; Must be 3 to qualify
	jc	outintr			; Not complete, get out
wade:
	mov	ebx,[port]		; Read a char
	os	SP_GETCHAR		;
	test	al,40h			; Sync bit set?
	loopz	wade			; Loop while not
	cmp	ecx,2			; See if at least two more bytes
	jc	outintr			; No, get out
	movzx	esi,al			; Mouse Buttons & high bits
	os	SP_GETCHAR
	movzx	edx,al			; EDX = Mouse column
	os	SP_GETCHAR
	movzx	ebx,al			; EBX = Mouse row
	mov	eax,esi			; Get high two bits of column
	and	al,3			;
	shl	al,6                    ;
	or	dl,al   		; Mix in with low six bits
	mov	eax,esi			; Get high two bits of row
	and	al,0ch			;
	shl	al,4			;
	or	bl,al			; Mix in with row
	bt	edx,7			; See if sign extending column
	jnc	short noadjdh		;
	or	edx,0ffffff00h		; Yes, do it
noadjdh:
	bt	ebx,7			; See if sign extending row
	jnc	short noadjbh		;
	or	ebx,0ffffff00h		; Yes, do it
noadjbh:
	mov	eax,esi			; Get button info to low two bits of al
	shr	al,4			;
endif

	bt	eax,MB_LEFT             ; See if left button
	jnc	short leftoff		; No, turn off
	bts	DWORD PTR [state],MB_LEFT; Turn it on
	jc	short leftdone		; Branch if already on
	inc	[leftcount]		; Else increment count
	jmp	short leftdone		; Done with left
leftoff:
	btr	DWORD PTR [state],MB_LEFT; Turn off left button
	btr	DWORD PTR [state],MB_LEFTDEC; And decrement hold
leftdone:
	bt	eax,MB_RIGHT		; See if left button
	jnc	short rightoff		; No, turn off
	bts	DWORD PTR [state],MB_RIGHT; Turn it on
	jc	short rightdone		; Branch if already on
	inc	[rightcount]		; Else increment count
	jmp	short rightdone		; Done
rightoff:
	btr	DWORD PTR [state],MB_RIGHT; Turn off left button
	btr	DWORD PTR [state],MB_RIGHTDEC; And decrement hold
rightdone:
	mov	eax,edx			; Get column delta
	imul	[colinscale]		; Scale it
	add	[col],eax		; Update column
	mov	eax,ebx     		; Get row delta
	imul	[rowinscale]		; Scale it
	add	[row],eax		; Update row
	mov	eax,[maxrow]		; Get maxrow
	cmp	eax,[row]		; See if row too big
	jns	short norowadj1		; No, check for too small
	mov	[row],eax		; Yes, Max it out
norowadj1:
	bt	[row],31		; Check if row negative
	jnc	short norowadj2		; No, done with row
	mov	[row],0			; Yes, min it out
norowadj2:
	mov	eax,[maxcol]		; Get max column
	cmp	eax,[col]		; Check if column too big
	jns	short nocoladj1		;
	mov	[col],eax		; Max it out if so
nocoladj1:
	bt	[col],31		; Test if column too small
	jnc	short nocoladj2		;
	mov	[col],0			; Minimize it if so
nocoladj2:
	sub	ecx,2			; We got two more chars
	cmp	ecx,3			; See if three or more left
	jnc	wade			; Loop if so
outintr:
	pop	ebx
	call	CondShowMouse		; Show mouse
noredraw:
	pop	es
	pop	edi			;
	pop	esi			;
	pop	edx			;
	pop	ecx			;
	pop	ebx			;
nointr:
	ret
MouseIntr	ENDP	
;
; Set scale factors
;
SetScale	PROC	
	or	cl,cl			; Output scales can't be 0
	jnz	ss_noadj1		;
	inc	cl			;
ss_noadj1:				;
	or	dl,dl			;
	jnz	ss_noadj2		;
	inc	dl			;
ss_noadj2:
	mov	BYTE PTR [rowinscale],ch; Load scale factors
	mov	BYTE PTR [rowoutscale],cl;
	mov	BYTE PTR [colinscale],dh;
	mov	BYTE PTR [coloutscale],dl;
	clc
	ret
SetScale	ENDP	
;
; Get scale factors
;
GetScale	PROC	
	mov	ch,BYTE PTR [rowinscale]; Load scale factors
	mov	cl,BYTE PTR [rowoutscale];
	mov	dh,BYTE PTR [colinscale];
	mov	dl,BYTE PTR [coloutscale];
	clc
	ret
GetScale	ENDP	
;
; Get position and press information
;
GetState	PROC	
	mov	eax,[row]		; Scale the row into ECX
	sub	edx,edx			;
	div	[rowoutscale]		;
	mov	ecx,eax			; Scale the column into EDX
	mov	eax,[col]		;
	sub	edx,edx			;
	div	[coloutscale]		;
	mov	edx,eax			; Assume no button presses
	sub	eax,eax
	bt	DWORD PTR [state],MB_LEFTDEC ;Are we in a hold state?
	jc	short gs_leftheld	; Yeah, mark it
	bt	DWORD PTR [state],MB_LEFT; Is Left on
	jc	short gs_leftheld2	; Yes, Decrement count
	test	[leftcount],-1		; Is anything in counter
	jz	short gs_leftnone	; No, no left button
	dec	[leftcount]		; Else dec counter
	jmp	short gs_left		; We have unheld left
gs_leftheld2:
	bts	DWORD PTR [state],MB_LEFTDEC ; See if already decremented for this hold state
	jc	gs_leftheld		; Go mark if so
	dec	[leftcount]		; Decrement
gs_leftheld:
	bts	eax,MB_LEFTHELD		; Left button is being held
gs_left:                                                           	
	bts	eax,MB_LEFT		; Left button is pressed
gs_leftnone:
	bt	DWORD PTR [state],MB_RIGHTDEC; In a hold state?
	jc	short gs_rightheld	; Branch if so
	bt	DWORD PTR [state],MB_RIGHT; Right button down
	jc	short gs_rightheld2	; Branch if so
	test	[rightcount],-1		; See if anything in count
	jz	short gs_rightnone	; No bits set if not
	dec	[rightcount]		; Else decrement count
	jmp	short gs_right		;
gs_rightheld2:
	bts	DWORD PTR [state],MB_RIGHTDEC; See if already decremented
	jc	gs_rightheld		; Yes, just set bits
	dec	[rightcount]		; Decrement count
gs_rightheld:
	bts	eax,MB_RIGHTHELD	; Being held
gs_right:
	bts	eax,MB_RIGHT		; Been pressed
gs_rightnone:
	clc
	ret
GetState	ENDP	
;
; Get the graphics cursor address, Text mode
;
TextCursorAddress	PROC	
	mov	eax,[row]		; Row/16 * chars/row + col/8  *2
	shr	eax,4
	movzx	ebx,[chars]		;
	mul	ebx			;
	mov	ebx,[col]		;
	shr	ebx,3			;
	add	eax,ebx			;
	add	eax,eax			;
	add	eax,[ScreenBase]	;
	mov	edi,eax			;
	ret
TextCursorAddress	ENDP	
;
; Get the Graphics Cursor Address, Graphics Mode
;   Also load number of lines, number of cols, and mask offset
;
GraphicsCursorAddress	PROC	
	mov	eax,[row]		; ROW
	mov	ch,CURSORLINES		; CH = lines to loop through
	mov	cl,CURSORCOLS		; CL = cols to loop through
	sub	esi,esi			; ESI = starting cursor bitmask offset
	mov	ebx,CharsPerLine	; EBX = chars/line
	mul	bx			; EAX = row * chars/line
	shl	edx,16			; Extend to 32 bits
	or	eax,edx			;
	sub	eax,[uhot]		;    - hotspot
	jns     gca_fulllines		; If positive, draw whole thing
gca_lp:
	dec	ch			; Else dec lines to draw
	add	esi,CURSORCOLS		; Point to next line in bitmap
	add	eax,ebx			; Next line
	js	gca_lp			; Loop whiel below screen
gca_fulllines:
	mov	ebx,[maxcol]		; Get max col
	inc	ebx			;
	sub	ebx,[col]		; Subtract col/8+1
	shr	ebx,3			;
	inc	ebx			;
	cmp	bl,cl			; See if clip at right
	jnc	short gca_allcols	; No, do all cols
	mov	cl,bl			; Else get cols to do
gca_allcols:
	mov	ebx,[col]		; Get col
	shr	ebx,3
	add	eax,ebx			; Add row and col
	mov	edi,eax			; ESI gets screen address
	add	edi,GRAPHBASE		;
	mov	eax,[col]	; See if need to shift left by columns
	sub	eax,[lhot]	;
	js	short gca_nohot	; NO, continue
	mov	eax,[col]	; Find out how far to shift
	and	eax,7		;
	sub	eax,[lhot]	;
	jns	gca_nohot	;
gca_hot:
	dec	edi		; Shift left one column
	inc	cl       	; Inc amount to claim
	add	eax,8		; See if done
	js	short gca_hot	;
gca_nohot:
	cmp	cl,3
	jc	short gca_ok
	mov	cl,3
gca_ok:
	ret
GraphicsCursorAddress	ENDP	
;
; Load hot spot
;
LoadHotSpot	PROC	
	and	edx,0f0fh		; Hot spot must be in pattern
	mov	eax,edx			;
	mov	BYTE PTR [lhot],ah	;
	mov	cl,CharsPerLine		;
	mul	cl			;
	mov	WORD PTR [uhot],ax	;
	ret
LoadHotSpot	ENDP	
;
; Copy a graphics mask
;
CopyMask	PROC	
	mov	ecx,CURSORLINES		; Number of lines
cm_lp:
	movsw				; Put pattern
	stosb				; Append third byte of mask
	loop	cm_lp			; Loop through whole pattern
	ret
CopyMask	ENDP	
;
; Copy a Graphics cursor to local memory
;
CopyGraphicsCursor	PROC	
	push	ds			; DS = user data seg
	push	fs			;
	pop	ds			;
	mov	edi,offset dgroup:grcursand	; Cursor position
	mov	al,0ffh			; AND mask has a third byte 0ffh
	call	CopyMask		; Copy AND mask
	sub	al,al			; XOR mask has a thrid byte 0
	call	CopyMask		; copy XOR mask
	pop	ds			; Restore data seg
	ret
CopyGraphicsCursor	ENDP	
;
; Save memory under cursor
;
SaveVideo	PROC	
	call	GraphicsCursorAddress	; Get cursor address
	push	ds			; DS = ABS data
	push	DSABS			;
	pop	ds			;
	mov	esi,edi			; ESI = screen buffer
	mov	edi,offset dgroup:overwritebuffer; EDI = overwrite buffer
	GETGCR				; Get the GCR
	mov	ah,3			; Plane 3
	mov	bl,cl			; bl = cols
sv_plane:
	GRAPHICSPLANE			; Select the graphics plane
	push	ecx			; Save rows
	push	esi			; Save screen buffer
sv_row:
	mov	cl,bl			; Get cols
	push	esi			; Save row ptr
sv_col:
	movsb				; Move a byte
        dec	cl			; Next col
	jnz	sv_col			;
	pop	esi			; Restore row ptr
	add	esi,CharsPerLine	; Next row
	dec	ch			;
	jnz	sv_row			;
	pop	esi			; Restore screen buffer
	pop	ecx			; Restore rows
	dec	ah			; Next plane
	jns	sv_plane		;
	pop	ds			; Restore DS
	ret
SaveVideo	ENDP	
;
; Restore memory under cursor
;
RestoreVideo	PROC	
	call	GraphicsCursorAddress	; Get cursor
	push	es			; ES = DSABS
	push	DSABS			;
	pop	es			;
	mov	esi,offset dgroup:overwritebuffer; Get save buffer
	GETGCR				; Load GCR
	RDMODE1				; Put us in Read mode 1
	mov	ah,8			; Plane 3
	mov	bl,cl			; Columns in bl
	GETSCR				; Get Sequencer
rs_plane:
	SEQUENCERPLANE			; Set the plane at the sequencer
	push	edi			; Save screen address
	push	ecx			; And rows
rs_row:
	mov	cl,bl			; Load cols
	push	edi			; Save current row
rs_col:
	movsb				; Move a byte
	dec	cl			; Next col
	jnz	rs_col			;
	pop	edi			; Restore current row
	add	edi,CharsPerLine	; Got to next row
	dec	ch			; Next row
	jnz	rs_row			;
	pop	ecx			; Restore rows
	pop	edi			; Restore screen address
	shr	ah,1			; Next plane
	jnc	rs_plane		; Loop through planes
	SPLANERESET			; Reset sequencer for all planes
	GETGCR				; Load graphics controller
	WRMODE0				; Read mode 0
	pop	es			; Restore es
	ret
RestoreVideo	ENDP	
;
; Apply a graphics cursor mask
;
MaskCursor	PROC	
	cld			; This CLD must be here in case interrupted an STD
mc_row:
	lodsb			; Get the three byte mask
	mov	bh,al		;
	lodsb			;
	mov	ah,al		;
	lodsb			;
	push	ecx		; See if undeflow
	pushfd			;
	mov	ecx,[col]	;
	sub	ecx,[lhot]	;
	js	short mc_leftshift
	and	ecx,7		;
	popfd			; Restore what to shift in from left
	jecxz	mc_noshift	; Don't shift if none
mc_rightshift:
	pushfd			; Save shift bit
	rcr	bh,1		; Do one shift right
	rcr	ax,1		;
	popfd			; Restore shift bit
	loop	mc_rightshift	; shift
	jmp	short mc_noshift
mc_leftshift:
	neg	ecx		; Negate shift count
	popfd			; Restore bit to shift in
mc_shiftleft:
	pushfd			; Save shift bit
	rcl	ax,1		; Do one left shift
	rcl	bh,1		;
	popfd			; Restore shift bit
	loop	mc_shiftleft	; More shift
mc_noshift:
	pop	ecx		; Restore row & col shifts
	pushfd			; Save shift bit
	mov	cl,bl		; Get cols
	push	edi		; Save screen pos
	push	eax		; Save bit pattern
	mov	al,es:[edi]	; Apply mask to first byte
	mov	es:[edi],bh	;
	pop	eax		;
	inc	edi		; Second byte
	dec	cl		; See if more than one byte to display
	jz	short mc_donerow; Quit if not
	mov	bh,es:[edi]	; Apply second byte of mask
	mov	es:[edi],ah	;
	inc	edi		; Point to third byte
	dec	cl		; See if to do a third byte
	jz	short mc_donerow; If not
	mov	bh,es:[edi]	; Apply third byte
	mov	es:[edi],al	;
mc_donerow:
	pop	edi		; Point to next row
	add	edi,CharsPerLine;
	popfd			; Restore shift bit
	dec	ch		; Next row
	jnz	mc_row		;
	ret
MaskCursor	ENDP	
; Draw the graphics cursor
;
DrawCursor	PROC	
	call	GraphicsCursorAddress	; Get cursor address
	push	es			; ES = absolute seg
	push	DSABS			;
	pop	es			;
	GETGCR				; Get control reg
	mov	bl,cl			; bl = cols
	push	edi			; Save regs
	push	ebx			;
	push	ecx			;
	push	esi                     ;
	add	esi,offset dgroup:grcursand	; Cursor mask
	WRFUNCAND			; Anding to bit planes
	stc				; Shift in a one bit
	call	MaskCursor		; Apply the and mask
	pop	esi
	pop	ecx			; Restore position
	pop	ebx			;
	pop	edi			;
	add	esi,offset dgroup:grcursxor	; Cursor mask
	WRFUNCXOR			; XORing to bit planes
	clc				; Shift in a 0 bit
	call	MaskCursor		; Apply the xor mask
	pop	es			; Restore absolute seg
	WRFUNCREPLACE			; Put us back in replace mode
	ret
DrawCursor	ENDP	
;
; Hide the mouse
;
HideMouse	PROC	
	sub	eax,eax
	btr	[inuse],FL_SHOWN	; See if shown
	jnc	short nohide		; Get out if not
	bts	eax,0
	push	eax
	test	[theGraphicsMode],1	; See if graphics op
	jnz	short HideGraphics	; Go do if so
	push	es
	push	DSABS
	pop	es
	call	TextCursorAddress	; Get cursor address
	mov	ax,WORD PTR [overwritebuffer]	; Get original contents
	mov	es:[edi],ax		; Write old value
	pop	es
	jmp	short finhide		; Get out
HideGraphics:
	call	RestoreVideo		; Restore the video buffer
finhide:
	pop	eax
nohide:
	clc
	ret
HideMouse	ENDP	
;
; Show the mouse
;
ShowMouse	PROC	
	bt	[inuse],FL_INUSE	; See if mouse in use
	jnc	short noshow		; Get out if not
	bts	[inuse],FL_SHOWN	; Set shown flag
	jc	short noshow		; Get out if already set
	test	[theGraphicsMode],1	; See if graphics mode
	jnz	short ShowGraphics	; Graphics op if so
	push	es
	push	DSABS
	pop	es
	call	TextCursorAddress	; Else get text cursor

	mov	ax,es:[edi]		; Original value to buffer
	mov	WORD PTR [overwritebuffer],ax	;
	rol	ah,4			; Swap attribs
	btr	eax,15			; Kill blink bit
	mov	es:[edi],ax		; Put value back
	pop	es
	jmp	short finshow		; get out
ShowGraphics:
	call	SaveVideo		; Save the video buffer
	call	DrawCursor		; Draw the cursor
finshow:
noshow:
	clc
	ret
ShowMouse	ENDP	
;
; Show mouse if bit 0 of eax is set
;
CondShowMouse	PROC	
	bt	ebx,0
	jc	ShowMouse
	ret
CondShowMouse	ENDP	
;
; Load the graphics cursor with user specified map
;
SetGraphicsCursor	PROC	
	call	LoadHotSpot		; Load hot spot
	call	CopyGraphicsCursor	; Copy the new cursor in
	clc
	ret
SetGraphicsCursor	ENDP	
;
; Mouse handler
MouseHandler	PROC	
	push	ebx			; Push EBX
	push	esi
	push	edi
	push	ds			;
	push	es
	push	fs

	push	ds			; FS = user data seg
	pop	fs

	push	DS386			; DS = data seg
	push	DS386			; ES = data seg
	pop	ds                      ;
	pop	es

	push	0			; Dispatch function
	call	TableDispatch
	dd	8
	dd	AttachMouse
	dd	DetachMouse
	dd	HideMouse
	dd	ShowMouse
	dd	CondShowMouse
	dd	SetScale
	dd	GetScale
	dd	GetState
	dd	SetGraphicsCursor
	pop	fs
	pop	es
	pop	ds			;
	pop	edi
	pop	esi
	pop	ebx			;
	ret
MouseHandler	ENDP	
seg386	ENDS	
END