;
; planes.asm
;
; this file does everything related to plane motion and error checking
;
; it initializes the plane structures, detects the one-second delay
; moves planes, checks for bounds/exits/conflicts
; decrements fuel and checks for out of fuel
; it also draws the right=hand info panel once each minute
;
ifdef SMALL
	.model small
else
	.model tiny
endif
	
	public	initplanes, moveplanes, movecheck, countplanes
	public timeleft,dist,check_conflict, clearplane

include atc.asi
include atc.mac
include rand.ase
include draw.ase
include fix.ase
include error.ase

	.data
timeleft dw	?	; time left in seconds
three dw	3     	; value three
sixty	dw	60	; minutes
lastime	dw	?	; last tick count we moved at
diradd  db	-1,0,-1,1,0,1,1,1,1,0,1,-1,0,-1,-1,-1


	.code
;
; random fix
;
; returns:
;  ax = fix (1-14, excluding navaids)
;
sdrand 	PROC
	mov	ax,12	; twelve non-navaid fixes
	call	rand
	cmp	al,10	; now adjust the last two into the airport position
	jc	sdrx
	add	al,2
sdrx:
	ret
sdrand	ENDP

;
; pythagorean theorem
;
; input:
;   si and di point to a row, column pair
;
; returns:
;   ax = result
;
dist	PROC
	push	bx
	push	cx
	push	di
	mov	al,[di].frow	; square of first
	sub	al,[si].frow
	jnc	noabs
	neg	al
noabs:
	mul	al
	push	ax		; save first
	mov	al,[di].fcol	; square of second
	sub	al,[si].fcol
	jnc	noabs2
	neg	al
noabs2:
	mul	al
	pop	di	; get first back
	add	di,ax	; di = value to take square root of
;
; a rather straightforward square root routine adapted from
; deja news.  Looks like the by-hand algorithm.  This is probably the
; best combination of small and fast I could find in the newsgroups...
;
; note : the square MUST fit in ten bits or you get bogus results.
; that is, if the playing field gets bigger fix the square root routine
; for more bits
;
    	xor 	dx,dx	; error
   	sub 	ax,ax	; answer
   	mov 	ch,5 	; 10-bit square to 5-bit square root
sqlp:
    	mov 	bx,di	; get next two bits of square
	mov	cl,8
    	shr 	bx,cl
	mov	cl,2
    	shl 	di,cl 	; ready for next pass
    	shl 	dx,cl 	; update our error with the bit pair
    	add 	dx,bx
   	add 	al,al	; make room for another bit and assume zero
   	mov 	bl,al   ; bx = 2*current result+1
   	add 	bl,bl
   	inc 	bl
    	cmp 	dx,bx	; if the error is greater than or equal to bx
   	jb  	sqc
   	inc 	al      ; the current bit is one
    	sub 	dx,bx   ; and lower the error
sqc:
	dec	ch
	jnz	sqlp	; and next pass
	pop	di
	pop	cx
	pop	bx
	ret
dist	ENDP
;
; erase all the plane info area
;
clearplane	PROC
	mov	di,PLANESTRUCS
	mov	cx,13 * size planes
	sub	ax,ax
	rep	stosw
	ret

clearplane	ENDP
;
; calculate entry times
;
; this is a subroutine primarily because the loop in initplanes
; went out of range
;
; input:
;   bx point to plane struct to initialize
;
; output:
;   [bx].secs and [bx].mins] set
;
caltime	PROC
	mov	ax,[timeleft]	; calculate entry time
	dec	ax		; adjust for the 1-sec start period
	sub	ax,(BASETIME-1) *60	; but subtract out the 15 free mins
	call	rand			;
	add	ax,(BASETIME-1)*60	; add back in the free stuff
	mov	[bx].secs,ax

	sub	dx,dx		; now calculate starting minute for
				; screen update
	div	[sixty]
	mov	[bx].mins,al

	add	bx,size planes	; point at next plane
	ret
caltime ENDP

;
; calculate the amount of fuel we need
;
; input:
;   bx = pointer to plane
;
; output:
;   [bx].fuel set
;
calfuel	PROC
	mov	dl,10		; lots of fuel if going out the way we came in
	cmp	si,di
	jz	randfueladd
	call	dist		; now get the distance between the source
				; and dest
	mov	dl,al		; dist in bl

randfueladd:
	mov	ax,10		; add up to ten minutes to the fuel
	call	rand
	add	dl,al		
	inc	dl              ; and throw in a free minute
	mov	[bx].fuel,dl
	ret
calfuel	ENDP
;
; make the planes struc
;
initplanes PROC
	mov	[lastime],-1
	mov	bx,PLANESTRUCS	; offset of planes strucs
	mov	cx,26		; number of planes
ipl1:
	mov	[bx].flags,0	; no flags (notably assume prop)
	mov	[bx].rotate,0 ; rotate is zero
	mov	ax,2		; now see if jet or prop
	call	rand
	or	al,al
	jz	isprop		; 0 = prop, continue
	mov	[bx].flags,F_JET ; flag jet
isprop:

	mov	al,cl		; set name
	neg	al
	add	al,'Z'+1
	mov	[bx].pname,al

	call	sdrand		; set up the destination
	mov	[bx].dest,al
	mul	[three]		; index into fixes tab
	mov	di,[fixptr]	; get a pointer to the fix data for dest
	mov	si,di
	add	di,ax

	call	sdrand		; get source fix
	mov	[bx].src,al
	sub	dl,dl		; assume not airport
	cmp	al,10		; see if is airport
	jc	ipna
     	mov	dl,4		; airport, reverse direction
ipna:
	push	dx
	mul	[three]
	pop	dx
	add	si,ax		; si = pointer to source
	cmp	ax,3*10		; see if source is airport
	mov	ax,1		; inc case yes, alt = 1
	jnc	airalt		; branch if is airport
	mov	ax,5		; else altitude is from five to nine
	call	rand
	add	al,5
airalt:
	mov	[bx].alt,al
	xor	dl,[si].fhead   ; load initial heading from source
	mov	[bx].head,dl
	mov	ax,word ptr [si].frow ; load initial location
	mov	word ptr [bx].row,ax

	call	calfuel     	; calculate the fuel

	call	caltime		; calculate entry times
	loop	ipl1
	ret
initplanes ENDP
;
; check if it is time for a move
;
; reads bios tick count and compares against last time we moved
;
; 18 ticks/second is good enough for DOS, so it is good enough for us.
;
; returns:
;   CY clear if time to move
;
movecheck PROC
	call	getticks		; get ticks
	sub	ax,[lastime]		; get last move time
	cmp	ax,18			; see if a second elasped
	jc	nomove
	add 	[lastime],ax
	clc
nomove:
	ret
movecheck ENDP
;
; move the planes if they need it
;
moveplanes PROC
	dec	[timeleft]			; next second
	jnz	notend
	sub	ax,ax         			; error if no time left
       	error	"Time ran out"
notend:
	mov	ax,[timeleft]			; get minutes in ax
	sub	dx,dx	
	div	[sixty] 			; dx = zero if new update needed
	or	dx,dx
	jnz	notnew
	call	drawnewplanes
	call	decfuel
notnew:
	mov	ax,[timeleft]			; time left
	mov	cx,26				; 26 planes
	mov	si,PLANESTRUCS			; plane data
mpl:
	cmp	ax,[si].secs	; see if time to draw
	jnz	mpn
	push	ax
	call	moveaplane			; yes, move the thing
	call	chghead
	pop	ax
mpn:
	add	si,size planes
	loop	mpl
	ret
moveplanes ENDP
;
; move a plane
;
; input:
;   si = pointer plane
;
moveaplane PROC
	call	check_exit			; check for exit conditions
	jc	mapquit
	test	[si].flags,F_ONSCREEN 		; see if new plane
	jz	redraw			; yes, draw it
	call	draw_bg				; else erase it
	test 	[si].flags,F_CIRCLE1 OR F_CIRCLE2 ; check circling
	jnz	circling
noheadchg:
	test	[si].rotate,7			; check rotating
	jz	moveit
	call	rotating
moveit:
	mov	bl,[si].head			; move the plane
	call	indexplane
nexttime:
	call	redraw
	call	check_bounds			; see if still in bounds
	call	check_conflict			; see if any conflicts
	ret
redraw:
	mov	ax,[timeleft]           	; calculate next update time
	sub	ax,15				; for jet
	test	[si].flags,F_JET
	jnz	mptx
     	sub	ax,15         			; no prop
mptx:
	mov	word ptr [si].secs,ax		; save next update time
	or	[si].flags,F_ONSCREEN		; visible now
	call	draw_plane			; draw plane
mapquit:
	ret
;
; rotater
;
rotating:
	dec	byte ptr [si].rotate	; dec count
rotatenodec:
	mov	bl,[si].head
	call	doturn
	mov	[si].head,bl
	ret
;
; circler
;
circling:
	mov	al,10				; assume navaid 1
	test	[si].flags,F_CIRCLE1		; first navaid?
	jnz	fnv
	inc	al    				; no, second fix
fnv:
	call	fixpos       			; get navaid fix
	mov	di,bx				;
	push	word ptr [si].row		; save old pos
	call	indexplane			; temp mov to see new dist
	add	si,planes.row			; find dist to navaid
	call	dist
	sub	si,planes.row
	cmp	al,[si].cirdist			; should move closer?
	pop	word ptr [si].row		; no, move straight
	jbe	cirnoturn         		; check if new dist too great
	call	rotatenodec			; yes, rotate once
cirnoturn:
	jmp	moveit				; no, go straight
moveaplane ENDP
;
; calculate new heading for turn
;
; input:
;   bl = old heading
;   si = planes ptr
;
; returns:
;   bl = new heading
;
doturn	PROC
	test	[si].flags,F_RIGHT
	jz	decdir
	inc	bl			; rotate right
	jmp	short dirx
decdir:
	dec	bl			; rotate left
dirx:
	and	bl,07h			; keep rotate in range
	ret
doturn	ENDP
;
; index the plane from one coord to next
;
; input:
;   si = ptr to plane
;   bl = heading
;
indexplane PROC
	mov	ax,word ptr [si].row		; get row,col
	sub	bh,bh
	shl	bx,1
	mov	ax,word ptr [bx +diradd]
	add	[si+planes.row],al
	add	[si+planes.col],ah
	ret
indexplane ENDP
;
; check if left field properly or not
;
; if the plane does NOT leave properly this routine will
; return with the carry clear, and, most of the time this
; will mean the plane keeps going and gets a bounds error
;
; input:
;   si = planes ptr
;
; return:
;   CY if plane is now gone
;
check_exit	PROC
	mov	al,[si].dest		; get dest position
	call	fixpos
	cmp	ax,word ptr [si].row	; see if is position
	jnz	noexit			; yes, chk alt
	cmp	[si].dest,10		; see if airport is dest
	jnc	destair
	cmp	[si].alt,5		; normal fix, alt must be 5 or greater
	jnc	exit2
	jmp	noexit
destair:
	cmp	[si].alt,0		; must have alt 0 to land
	jnz	noexit
	test	[si].flags,F_CTL	; must have clear to land to land
	jnz	exit2
noexit:
	clc
	ret
exit2: 
	mov	al,[bx].fhead		; get the heading for this fix
	xor	al,4			; convert to exit direction
	cmp	al,[si].head		; see if right heading
	jnz	noexit			; quit if not
	call	draw_bg			; valid exit cond, get rid of plane
	and	[si].flags, NOT F_ONSCREEN ; delete plane
	mov	[si].secs,-256		; mark it out successfully
	stc
	ret
check_exit	ENDP
;
; check if left field improperly or overflew an airport at low alt
;
; input:
;   si = planes ptr
;
check_bounds	PROC
	mov	ax,word ptr [si].row	; check if on board
	or	al,al
	jl	badbounds
	or	ah,ah
	jl	badbounds
	cmp	al,ROWS
	jnc	badbounds
	cmp	ah,COLS
	jnc	badbounds
	mov	al,12 			; now check first airport
	cmp	al,[si].dest		; but not if is dest
	jz	cbnotair1
	call	fixpos			; get position
	cmp	ax,word ptr [si].row	; are we there
	jz	boundsair		; yes, check alt
cbnotair1:
	mov	al,13			; second airport
	cmp	al,[si].dest		; but not if is dest
	jz	cbnotair2
	call	fixpos			; get pos
	cmp	ax,word ptr [si].row	; are we there
	jz	boundsair		; yes, check alt
cbnotair2:
	ret
;
; over an airport which is not the dest
;
boundsair:
	cmp	[si].alt,4		; altitude must be 4 or greater
	jc	badbounds
	ret
	
;
; out of bounds
;
badbounds:
	sub	ah,ah     	
	mov	al,[si].pname   	; get plane name
	error	"Plane out of bounds - "; error
check_bounds	ENDP
;
; put up message if no more planes
;
quitifdone PROC
	call	countplanes		; count planes
	or	al,al
	jz	go
	ret
go:
	sub	ax,ax
	error	"Game over, you win!"
quitifdone ENDP
;
; count the number of planes which are left
;
; returns
;   ax : plane count
;
countplanes PROC
	sub	ax,ax			; count in ax
	push	cx
	mov	cx,26			; 26 planes
	mov	di,PLANESTRUCS		; plane strucs
cpnl:
	cmp	[di].secs,-256	;l see if done
	jz	cpni
     	inc	ax			; no, inc
cpni:
	add	di,size planes		; next plane
	loop	cpnl
	pop	cx
	ret
countplanes ENDP
;
; check for conflicts
;
check_conflict PROC
	add	si,planes.row
	push	cx
	mov	cx,26			; 26 planes
	mov	di,PLANESTRUCS + planes.row	; plane strucs
	
cfnl:
	cmp	si,di			; don't check against self
	jz	ncf
	test	[di-planes.row + planes.flags],F_ONSCREEN ; or against off-screen planes
	jz	ncf
	mov	al,[si-planes.row +planes.alt] ; see if same alt
	cmp	al,[di-planes.row +planes.alt] ; see if same alt
	jnz	ncf
	call	dist			; same alt, check dist
	cmp	al,4
	jnc	ncf
	sub	si,planes.row		; conflict, display error
	sub	di,planes.row
	mov	ah,[di].pname
	mov	al,[si].pname
	error	"Conflict between "
ncf:
	add	di,size planes		; next plane
	loop	cfnl
	pop	cx
	sub	si,planes.row
	ret
check_conflict ENDP

;
; decrement fuel counts of active planes
; and gen an error if one runs out of fuel
decfuel PROC
	push	cx
	mov	cx,26			; 26 planes
	mov	di,PLANESTRUCS		; plane strucs
dfnl:
	test	[di.flags],F_ONSCREEN
	jz	fnd
	dec	[di].fuel
	jz	ferr
fnd:
	add	di,size planes		; next plane
	loop	dfnl
	pop	cx
	ret
ferr:
	sub	ax,ax
	mov	al,[di].pname
	error	"Out of fuel - "

decfuel ENDP
;
; change the heading if over a navaid and the head change flag is on
;
; input
;   si = planes ptr
;
chghead PROC
	test	[si].flags,F_CHGHEAD ; see if head change flag set
	jz	noxheadchg
	mov	al,10			; see if over a navaid
	call	fixpos
	cmp	ax,word ptr [si].row
	jnz	notair1
	mov	al,12
	jmp	fixgot
notair1:
	mov	al,11			; see if over a navaid
	call	fixpos
	cmp	ax,word ptr [si].row
	jnz	noxheadchg
	mov	al,13
fixgot:
	call	fixpos
	mov	al,[bx].fhead
	xor	al,4
	mov	[si].head,al
	mov	[si].rotate,0
noxheadchg:
	ret
chghead ENDP
;
; draw the new planes info
;
; input:
;   ax = minute
;
drawnewplanes PROC
	push	ax			; stack = time
	call	draw_time		; display the time
	call	erase_info_region	; erase the info area
	pop	dx			; dl = minute-1
	dec	DL
	mov	ax,INFO_ROW + (INFO_COL SHL 8) ; ax = first plane pos
	sub	bx,bx			; bh = 0 planes drawn
					; bl = 0 for no command info
	mov	si,PLANESTRUCS		; pointer to planes
	mov	cx,26			; 26 planes
dnl:
	cmp	[si].mins,dl	; see if coming in this minute
	jnz	dn_nodraw
	push	ax			; display the plane info
	push	bx
	push	cx
	push	dx
	call	draw_plane_info
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	inc	bh			; add one to displ;ay count
	inc	al			; increment row
	cmp	bh,16			; see if done 16
	jnz	dn_nodraw
	mov	ax,INFO_ROW  + (INFO_COL+INFO_INDEX) SHL 8; yes, position to top of next row
dn_nodraw:
	add	si,size planes		; next plane
	loop	dnl
	ret
drawnewplanes ENDP
	end